/**
 * TinyMCE Custom Format Painter Plugin
 * Este plugin básico permite copiar a formatação (tag HTML e estilos inline)
 * de um texto e aplicá-la a outro.
 *
 * Limitações:
 * - Copia apenas a tag e estilos inline do elemento pai direto da seleção.
 * - Não é tão inteligente quanto um format painter premium (não lida com estilos CSS complexos, herança, etc.).
 * - Aplica a formatação envolvendo o novo texto selecionado com a tag e estilos copiados.
 */
tinymce.PluginManager.add('customformatpainter', function(editor, url) { // Alterado de 'formatpainter' para 'customformatpainter'

    let _copiedFormat = null; // Armazena a formatação copiada { tagName: '...', style: '...' }
    let _isActive = false;    // Indica se o pincel de formatação está ativo (modo copiar)
    let _buttonInstance = null; // Referência para a instância do botão para atualizar seu estado

    // Função para atualizar o estado visual do botão
    const updateButtonState = () => {
        if (_buttonInstance) {
            _buttonInstance.setActive(_isActive);
            _buttonInstance.setTooltip(_isActive ? 'Aplicar Formato (Clique novamente para desativar)' : 'Pincel de Formatação');
        }
    };

    // Adiciona o botão à barra de ferramentas como um ToggleButton
    editor.ui.registry.addToggleButton('customformatpainter', { // Alterado de 'formatpainter' para 'customformatpainter'
        icon: 'format-painter', // Usa o ícone 'format-painter' se disponível
        tooltip: 'Pincel de Formatação',
        onAction: function (buttonApi) {
            // Lógica para alternar o estado do pincel ao clicar no botão
            if (!_isActive) {
                // MODO COPIAR FORMATO
                const selectedNode = editor.selection.getNode();
                const selectedContent = editor.selection.getContent({ format: 'html' });

                // Se nada for selecionado ou a seleção for o corpo do editor, não faça nada
                if (editor.selection.isCollapsed() || (selectedNode === editor.getBody() && selectedContent === '')) {
                    editor.notificationManager.open({
                        text: 'Selecione o texto cuja formatação você deseja copiar.',
                        type: 'info',
                        timeout: 3000
                    });
                    // Não ativa o pincel se não houver seleção válida
                    _isActive = false;
                    updateButtonState();
                    return;
                }

                // Captura o nome da tag e os estilos inline do nó selecionado
                let tagName = selectedNode.tagName.toLowerCase();
                let styleAttribute = selectedNode.getAttribute('style') || '';

                _copiedFormat = {
                    tagName: tagName,
                    style: styleAttribute
                };
                _isActive = true; // Ativa o pincel
                updateButtonState(); // Atualiza o estado visual do botão

                editor.notificationManager.open({
                    text: 'Formato copiado. Agora, selecione o texto onde deseja aplicar o formato e clique no pincel novamente.',
                    type: 'success',
                    timeout: 5000
                });

            } else {
                // MODO APLICAR FORMATO
                if (!_copiedFormat) {
                    // Deveria ter um formato copiado, mas por segurança
                    editor.notificationManager.open({
                        text: 'Nenhum formato foi copiado. Selecione um texto primeiro.',
                        type: 'warning',
                        timeout: 3000
                    });
                    _isActive = false; // Desativa o pincel
                    updateButtonState();
                    return;
                }

                const newSelectionContent = editor.selection.getContent({ format: 'html' });

                if (newSelectionContent === '') {
                    editor.notificationManager.open({
                        text: 'Nenhum texto selecionado para aplicar o formato.',
                        type: 'info',
                        timeout: 3000
                    });
                    return;
                }

                // Cria o novo HTML com a formatação copiada envolvendo o conteúdo selecionado
                let formattedHtml = `<${_copiedFormat.tagName} style="${_copiedFormat.style}">${newSelectionContent}</${_copiedFormat.tagName}>`;

                // Insere o conteúdo formatado, substituindo a seleção atual
                editor.execCommand('mceInsertContent', false, formattedHtml);

                // Resetar o estado após aplicar
                _copiedFormat = null;
                _isActive = false;
                updateButtonState(); // Atualiza o estado visual do botão

                editor.notificationManager.open({
                    text: 'Formato aplicado.',
                    type: 'success',
                    timeout: 3000
                });
            }
        },
        // A função setup é chamada quando o editor é inicializado
        // e recebe a instância do botão para gerenciamento de estado.
        setup: function (buttonApi) {
            _buttonInstance = buttonApi; // Armazena a referência para a instância do botão

            // Atualiza o estado inicial do botão
            updateButtonState();

            // Este ouvinte garante que o botão reflita o estado (_isActive)
            // mesmo após outras ações no editor que possam alterar a seleção.
            editor.on('NodeChange', function () {
                updateButtonState();
            });

            // Opcional: Desativar o pincel se o usuário clicar fora do editor ou perder o foco
            // editor.on('blur', function() {
            //     _isActive = false;
            //     _copiedFormat = null;
            //     updateButtonState();
            // });
        }
    });

    // Retorna informações sobre o plugin (opcional)
    return {
        getMetadata: function () {
            return {
                name: 'Custom Format Painter Plugin',
                url: 'https://seusite.com/customformatpainter' // Substituído para o novo nome do plugin
            };
        }
    };
});
